/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.wicket.markup.repeater;

import java.util.Comparator;

import org.apache.wicket.markup.html.WebMarkupContainer;
import org.apache.wicket.model.IModel;
import org.apache.wicket.version.undo.Change;


/**
 * Container that holds components in a RefreshingView. One Item represents one entire row of the
 * view. Users should add all containing components to the Item instead of the view, this is
 * accomplished by implementing refreshingView.populateItem(Item item).
 * 
 * @see RefreshingView
 * 
 * @author Igor Vaynberg (ivaynberg)
 * 
 * @param <T>
 *            Model object type
 */
public class Item<T> extends WebMarkupContainer
{
	private static final long serialVersionUID = 1L;

	/** relative index of this item */
	private int index;

	/**
	 * @param id
	 *            component id
	 * @param index
	 *            relative index of this item in the pageable view
	 * @param model
	 *            model for this item
	 */
	public Item(final String id, int index, final IModel<T> model)
	{
		super(id.intern(), model);
		this.index = index;
	}

	/**
	 * @param id
	 *            component id
	 * @param index
	 *            relative index of this item in the pageable view
	 */
	public Item(final String id, int index)
	{
		super(id.intern());
		this.index = index;
	}

	/**
	 * Sets the index of this item
	 * 
	 * @param index
	 *            new index
	 */
	public void setIndex(int index)
	{
		if (this.index != index)
		{
			if (isVersioned())
			{
				addStateChange(new Change()
				{
					final int oldIndex = Item.this.index;
					private static final long serialVersionUID = 1L;

					@Override
					public void undo()
					{
						Item.this.index = oldIndex;
					}

					@Override
					public String toString()
					{
						return "IndexChange[component: " + getPath() + ", index: " + oldIndex + "]";
					}
				});
			}
			this.index = index;
		}
	}

	/**
	 * @return the index assigned to this item
	 */
	public int getIndex()
	{
		return index;
	}


	/**
	 * @return the primary key assigned to this item
	 */
	public String getPrimaryKey()
	{
		return getId();
	}

	/**
	 * Comparator that compares Items by their index property
	 * 
	 * @author Igor Vaynberg (ivaynberg)
	 * 
	 */
	public static class IndexComparator implements Comparator<Item<?>>
	{
		private static final Comparator<Item<?>> instance = new IndexComparator();

		/**
		 * @return static instance of the comparator
		 */
		public static final Comparator<Item<?>> getInstance()
		{
			return instance;
		}

		/**
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		public int compare(Item<?> lhs, Item<?> rhs)
		{
			return lhs.getIndex() - rhs.getIndex();
		}

	};

	/**
	 * Gets model
	 * 
	 * @return model
	 */
	@SuppressWarnings("unchecked")
	public final IModel<T> getModel()
	{
		return (IModel<T>)getDefaultModel();
	}

	/**
	 * Sets model
	 * 
	 * @param model
	 */
	public final void setModel(IModel<T> model)
	{
		setDefaultModel(model);
	}

	/**
	 * Gets model object
	 * 
	 * @return model object
	 */
	@SuppressWarnings("unchecked")
	public final T getModelObject()
	{
		return (T)getDefaultModelObject();
	}

	/**
	 * Sets model object
	 * 
	 * @param object
	 */
	public final void setModelObject(T object)
	{
		setDefaultModelObject(object);
	}


}
