/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.wicket.markup.html.list;

import org.apache.wicket.markup.html.WebMarkupContainer;
import org.apache.wicket.model.IModel;

/**
 * Container that holds components in a ListView.
 * 
 * @see ListView
 * @author Jonathan Locke
 * 
 * @param <T>
 *            Model object type
 */
public class ListItem<T> extends WebMarkupContainer
{
	private static final long serialVersionUID = 1L;

	/** The index of the ListItem in the parent ListView */
	private final int index;

	/**
	 * A constructor which uses the index and the list provided to create a ListItem. This
	 * constructor is the default one.
	 * 
	 * @param index
	 *            The index of the item
	 * @param model
	 *            The model object of the item
	 */
	public ListItem(final int index, final IModel<T> model)
	{
		super(Integer.toString(index).intern(), model);
		this.index = index;
	}

	/**
	 * Gets the index of the listItem in the parent listView.
	 * 
	 * @return The index of this listItem in the parent listView
	 */
	public final int getIndex()
	{
		return index;
	}

	/**
	 * Gets model
	 * 
	 * @return model
	 */
	@SuppressWarnings("unchecked")
	public final IModel<T> getModel()
	{
		return (IModel<T>)getDefaultModel();
	}

	/**
	 * Sets model
	 * 
	 * @param model
	 */
	public final void setModel(IModel<T> model)
	{
		setDefaultModel(model);
	}

	/**
	 * Gets model object
	 * 
	 * @return model object
	 */
	@SuppressWarnings("unchecked")
	public final T getModelObject()
	{
		return (T)getDefaultModelObject();
	}

	/**
	 * Sets model object
	 * 
	 * @param object
	 */
	public final void setModelObject(T object)
	{
		setDefaultModelObject(object);
	}
}
